#!/usr/bin/env python

# Mostly copied the contents of this file from:
# https://github.com/kontron/python-ipmi
# Copyright (c) 2014  Kontron Europe GmbH
#
# Modified by: Puneet Shenoy <puneet@cumulusnetworks.com>
# Copyright (C) 2014 Cumulus Networks, Inc.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

try:
    import sys
    import array
    import datetime
    import eeprom_base
    import syslog
except ImportError, e:
    raise ImportError (str(e) + "- required module not found")


class DecodingError(Exception):
    """Error on message decoding."""
    pass

class Fru(eeprom_base.eepromDecoder):
    def __init__(self, eeprom_length, eeprom_path):
        super(Fru, self).__init__(eeprom_path, None, 0, '', False)
        self.eeprom_length = eeprom_length

    def get_fru_inventory_area_info(self):
        return self.eeprom_length

    def write_fru_data(self, data, offset=0):
        pass

    def read_fru_data(self):
        data = array.array('B', super(Fru, self).read_eeprom_bytes(self.eeprom_length))
        return data.tostring()

    def get_fru_inventory(self):
        return FruInventory(self.read_fru_data())

class FruDataField:
    TYPE_BINARY = 0
    TYPE_BCD_PLUS = 1
    TYPE_6BIT_ASCII = 2
    TYPE_ASCII_OR_UTF16 = 3

    def __init__(self, data=None, offset=0, force_lang_english=False):
        if data:
            self.from_data(data, offset, force_lang_english)

    def __str__(self):
        if self.type is FruDataField.TYPE_BINARY:
            return ' '.join('%02x' % ord(b) for b in self.value)
        else:
            return self.value.replace('\x00', '')

    def from_data(self, data, offset=0, force_lang_english=False):
        self.type = ord(data[offset]) >> 6 & 0x3
        self.length = ord(data[offset]) & 0x3f

        self.raw = data[offset+1:offset+1+self.length]

        if type == self.TYPE_BCD_PLUS:
            value = self.raw.decode('bcd+')
        elif type == self.TYPE_6BIT_ASCII:
            value = self.raw.decode('6bitascii')
        else:
            value = self.raw

        self.value = value

class InventoryCommonHeader:
    def __init__(self, data=None):
        if data:
            self.from_data(data)

    def from_data(self, data):
        if len(data) != 8:
            raise DecodingError('InventoryCommonHeader length != 8')
        self.format_version = ord(data[0]) & 0x0f
        self.internal_use_area_offset = ord(data[1]) * 8 or None
        self.chassis_info_area_offset = ord(data[2]) * 8 or None
        self.board_info_area_offset = ord(data[3]) * 8 or None
        self.product_info_area_offset = ord(data[4]) * 8 or None
        self.multirecord_area_offset = ord(data[5]) * 8 or None
        if sum([ord(c) for c in data]) % 256 != 0:
            self.checksum_fail = True

class CommonInfoArea:
    def __init__(self, data=None):
        if data:
            self.from_data(data)

    def from_data(self, data):
        self.format_version = ord(data[0]) & 0x0f
        if self.format_version != 1:
            raise DecodingError('unsupported format version (%d)' %
                    self.format_version)
        self.length = ord(data[1]) * 8
        if sum([ord(c) for c in data[:self.length]]) % 256 != 0:
            self.checksum_fail = True

class InventoryChassisInfoArea(CommonInfoArea):
    TYPE_OTHER = 1
    TYPE_UNKNOWN = 2
    TYPE_DESKTOP = 3
    TYPE_LOW_PROFILE_DESKTOP = 4
    TYPE_PIZZA_BOX = 5
    TYPE_MINI_TOWER = 6
    TYPE_TOWER = 7
    TYPE_PORTABLE = 8
    TYPE_LAPTOP = 9
    TYPE_NOTEBOOK = 10
    TYPE_HAND_HELD = 11
    TYPE_DOCKING_STATION = 12
    TYPE_ALL_IN_ONE = 13
    TYPE_SUB_NOTEBOOK = 14
    TYPE_SPACE_SAVING = 15
    TYPE_LUNCH_BOX = 16
    TYPE_MAIN_SERVER_CHASSIS = 17
    TYPE_EXPANSION_CHASSIS = 18
    TYPE_SUB_CHASSIS = 19
    TYPE_BUS_EXPANSION_CHASSIS = 20
    TYPE_PERIPHERAL_CHASSIS = 21
    TYPE_RAID_CHASSIS = 22
    TYPE_RACK_MOUNT_CHASSIS = 23

    def from_data(self, data):
        CommonInfoArea.from_data(self, data)
        self.type = ord(data[2])
        offset = 3
        self.part_number = FruDataField(data, offset)
        offset += self.part_number.length+1
        self.serial_number = FruDataField(data, offset, True)
        offset += self.serial_number.length+1
        self.custom_chassis_info = list()
        while ord(data[offset]) != 0xc1:
            field = FruDataField(data, offset)
            self.custom_chassis_info.append(field)
            offset += field.length+1

class InventoryBoardInfoArea(CommonInfoArea):
    def from_data(self, data):
        CommonInfoArea.from_data(self, data)
        self.language_code = ord(data[2])
        minutes = ord(data[5]) << 16 | ord(data[4]) << 8 | ord(data[3])
        self.mfg_date = (datetime.datetime(1996, 1, 1)
                + datetime.timedelta(minutes=minutes))
        offset = 6
        self.manufacturer = FruDataField(data, offset)
        offset += self.manufacturer.length+1
        self.product_name = FruDataField(data, offset)
        offset += self.product_name.length+1
        self.serial_number = FruDataField(data, offset, True)
        offset += self.serial_number.length+1
        self.part_number = FruDataField(data, offset)
        offset += self.part_number.length+1
        self.fru_file_id = FruDataField(data, offset, True)
        offset += self.fru_file_id.length+1
        self.custom_mfg_info = list()
        while ord(data[offset]) != 0xc1:
            field = FruDataField(data, offset)
            self.custom_mfg_info.append(field)
            offset += field.length+1

class InventoryProductInfoArea(CommonInfoArea):
    def from_data(self, data):
        CommonInfoArea.from_data(self, data)
        self.language_code = ord(data[2])
        offset = 3
        self.manufacturer = FruDataField(data, offset)
        offset += self.manufacturer.length+1
        self.name = FruDataField(data, offset)
        offset += self.name.length+1
        self.part_number = FruDataField(data, offset)
        offset += self.part_number.length+1
        self.version = FruDataField(data, offset)
        offset += self.version.length+1
        self.serial_number = FruDataField(data, offset, True)
        offset += self.serial_number.length+1
        self.asset_tag = FruDataField(data, offset)
        offset += self.asset_tag.length+1
        self.fru_file_id = FruDataField(data, offset, True)
        offset += self.fru_file_id.length+1
        self.custom_mfg_info = list()
        while ord(data[offset]) != 0xc1:
            field = FruDataField(data, offset)
            self.custom_mfg_info.append(field)
            offset += field.length+1


class FruDataMultiRecord:
    TYPE_POWER_SUPPLY_INFORMATION = 0
    TYPE_DC_OUTPUT = 1
    TYPE_DC_LOAD = 2
    TYPE_MANAGEMENT_ACCESS_RECORD = 3
    TYPE_BASE_COMPATIBILITY_RECORD = 4
    TYPE_EXTENDED_COMPATIBILITY_RECORD = 5
    TYPE_OEM = range(0x0c, 0x100)
    TYPE_OEM_PICMG = 0xc0

    def __init__(self, data):
        if data:
            self.from_data(data)

    def __str__(self):
        return '%02x: %s' % (self.type,
                ' '.join('%02x' % ord(b) for b in self.raw))

    def from_data(self, data):
        if len(data) < 5:
            raise DecodingError('data too short')
        self.record_type_id = ord(data[0])
        self.format_version = ord(data[1]) & 0x0f
        self.end_of_list = bool(ord(data[1]) & 0x80)
        self.length = ord(data[2])
        if sum([ord(c) for c in data[:5]]) % 256 != 0:
            self.checksum_fail = True
        self.raw = data[5:5+self.length]
        if (sum([ord(c) for c in self.raw]) + ord(data[3])) % 256 != 0:
            self.checksum_fail = True

    @staticmethod
    def create_from_record_id(data):
        if ord(data[0]) == FruDataMultiRecord.TYPE_OEM_PICMG:
            return FruPicmgRecord.create_from_record_id(data)
        else:
            return FruDataUnknown(data)


class FruDataUnknown(FruDataMultiRecord):
    """This class is used to indicate undecoded picmg record."""
    def __str__(self):
        return "Not found"


class FruPicmgRecord(FruDataMultiRecord):
    PICMG_RECORD_ID_BACKPLANE_PTP_CONNECTIVITY = 0x04
    PICMG_RECORD_ID_ADDRESS_TABLE = 0x10
    PICMG_RECORD_ID_SHELF_POWER_DISTRIBUTION = 0x11
    PICMG_RECORD_ID_SHMC_ACTIVATION_MANAGEMENT = 0x12
    PICMG_RECORD_ID_SHMC_IP_CONNECTION = 0x13
    PICMG_RECORD_ID_BOARD_PTP_CONNECTIVITY = 0x14
    PICMG_RECORD_ID_RADIAL_IPMB0_LINK_MAPPING = 0x15
    PICMG_RECORD_ID_MODULE_CURRENT_REQUIREMENTS = 0x16
    PICMG_RECORD_ID_CARRIER_ACTIVATION_MANAGEMENT = 0x17
    PICMG_RECORD_ID_CARRIER_PTP_CONNECTIVITY = 0x18
    PICMG_RECORD_ID_AMC_PTP_CONNECTIVITY = 0x19
    PICMG_RECORD_ID_CARRIER_INFORMATION = 0x1a
    PICMG_RECORD_ID_MTCA_FRU_INFORMATION_PARTITION = 0x20
    PICMG_RECORD_ID_MTCA_CARRIER_MANAGER_IP_LINK = 0x21
    PICMG_RECORD_ID_MTCA_CARRIER_INFORMATION = 0x22
    PICMG_RECORD_ID_MTCA_SHELF_INFORMATION = 0x23
    PICMG_RECORD_ID_MTCA_SHELF_MANAGER_IP_LINK = 0x24
    PICMG_RECORD_ID_MTCA_CARRIER_POWER_POLICY = 0x25
    PICMG_RECORD_ID_MTCA_CARRIER_ACTIVATION_AND_POWER = 0x26
    PICMG_RECORD_ID_MTCA_POWER_MODULE_CAPABILITY = 0x27
    PICMG_RECORD_ID_MTCA_FAN_GEOGRAPHY = 0x28
    PICMG_RECORD_ID_OEM_MODULE_DESCRIPTION = 0x29
    PICMG_RECORD_ID_CARRIER_CLOCK_PTP_CONNECTIVITY = 0x2C
    PICMG_RECORD_ID_CLOCK_CONFIGURATION = 0x2d
    PICMG_RECORD_ID_ZONE_3_INTERFACE_COMPATIBILITY = 0x30
    PICMG_RECORD_ID_CARRIER_BUSED_CONNECTIVITY = 0x31
    PICMG_RECORD_ID_ZONE_3_INTERFACE_DOCUMENTATION = 0x32

    def __init__(self, data):
        FruDataMultiRecord.__init__(self, data)

    @staticmethod
    def create_from_record_id(data):
        picmg_record = FruPicmgRecord(data)
        if picmg_record.picmg_record_type_id ==\
            FruPicmgRecord.PICMG_RECORD_ID_MTCA_POWER_MODULE_CAPABILITY:
            return FruPicmgPowerModuleCapabilityRecord(data)
        else:
            return FruPicmgRecord(data)

    def from_data(self, data):
        if len(data) < 10:
            raise DecodingError('data too short')
        FruDataMultiRecord.from_data(self, data)
        self.manufacturer_id = ord(data[5])|ord(data[6])<<8|ord(data[7])<<16
        self.picmg_record_type_id = ord(data[8])
        self.format_version = ord(data[9])


class FruPicmgPowerModuleCapabilityRecord(FruPicmgRecord):
    def from_data(self, data):
        if len(data) < 12:
            raise DecodingError('data too short')
        FruPicmgRecord.from_data(self,data)
        maximum_current_output = ord(data[10])|ord(data[11])<<8
        self.maximum_current_output = float(maximum_current_output/10)


class InventoryMultiRecordArea:
    def __init__(self, data):
        if data:
            self.from_data(data)

    def from_data(self, data):
        self.records = list()
        offset = 0
        while True:
            record = FruDataMultiRecord.create_from_record_id(data[offset:])
            self.records.append(record)
            offset += record.length+5
            if record.end_of_list:
                break


class FruInventory:
    def __init__(self, data=None):
        self.chassis_info_area = None
        self.board_info_area = None
        self.product_info_area = None

        if data:
            self.from_data(data)

    def from_data(self, data):
        self.raw = data
        self.common_header = InventoryCommonHeader(data[:8])

        if self.common_header.chassis_info_area_offset:
            self.chassis_info_area = InventoryChassisInfoArea(
                    data[self.common_header.chassis_info_area_offset:])

        if self.common_header.board_info_area_offset:
            self.board_info_area = InventoryBoardInfoArea(
                    data[self.common_header.board_info_area_offset:])

        if self.common_header.product_info_area_offset:
            self.product_info_area = InventoryProductInfoArea(
                    data[self.common_header.product_info_area_offset:])

        if self.common_header.multirecord_area_offset:
            self.multirecord_area = InventoryMultiRecordArea(
                    data[self.common_header.multirecord_area_offset:])

class IpmiFruDecoder(Fru):
    eeprom_size = 256

    def __init__(self, name, path, cpld_root, ro):
        super(IpmiFruDecoder, self).__init__(self.eeprom_size, path)
        self.name = name

    def check_status(self):
        return 'absent'

    def switchaddrstr(self, inv):
        return 'No switch address.'

    def mgmtaddrstr(self, inv):
        return 'No switch mgmt address.'

    def switchaddrrange(self, inv):
        return 'No switch address range.'

    def serial_number_str(self, inv):
        return 'No switch serial number.'

    def is_read_only(self):
        # readonly for now.
        return 1

    def is_checksum_valid(self, inv):
        return(True, 0)

    def read_eeprom(self):
        return self.get_fru_inventory()

    def update_cache(self, inv):
        super(IpmiFruDecoder, self).update_cache(inv.raw)

    def decode_eeprom_dictionary(self, inv):
        info = {}
        if inv.chassis_info_area:
            info['chassis_info'] = self._encode_area(inv.chassis_info_area)
        if inv.board_info_area:
            info['board_info'] = self._encode_area(inv.board_info_area)
        if inv.product_info_area:
            info['product_info'] = self._encode_area(inv.product_info_area)
        return info

    def _encode_area(self, area):
        area_info = {}
        for key, val in area.__dict__.iteritems():
            area_info[key] = str(val)
        return area_info

    def decode_eeprom(self, inv):
        # Chassis Info Area
        area = inv.chassis_info_area
        if area:
            #
            # If for example the EEPROM for an unsupported PSU is read, or
            # a supported one is misprogrammed, this print may throw KeyError.
            # Under similar conditions the custom chassis info area reference
            # below may throw an AttributeError (cf. CM-19193).
            #
            try:
                print '''
  Chassis Info Area:
  Type:               %(type)d
  Part Number:        %(part_number)s
  Serial Number:      %(serial_number)s
'''[1:-1] % area.__dict__

            except KeyError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding keys in chassis info area')

            try:
                if len(area.custom_chassis_info) != 0:
                    print '  Custom Chassis Info Records:'
                    for field in area.custom_chassis_info:
                        print '    %s' % field
            except AttributeError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding attributes in custom chassis info area')

        # Board Info Area
        area = inv.board_info_area
        if area:
            try:
                print '''
Board Info Area:
  Mfg. Date / Time:   %(mfg_date)s
  Manufacturer:       %(manufacturer)s
  Product Name:       %(product_name)s
  Serial Number:      %(serial_number)s
  Part Number:        %(part_number)s
  FRU File ID:        %(fru_file_id)s
'''[1:-1] % area.__dict__

            except KeyError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding keys in board info area')

            try:
                if len(area.custom_mfg_info) != 0:
                    print '  Custom Board Info Records:'
                    for field in area.custom_mfg_info:
                        print '    %s' % field
            except AttributeError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding attributes in custom mfg info area (board)')

        # Product Info Area
        area = inv.product_info_area
        if area:
            try:
                print '''
Product Info Area:
  Manufacturer:       %(manufacturer)s
  Name:               %(name)s
  Part/Model Number:  %(part_number)s
  Version:            %(version)s
  Serial Number:      %(serial_number)s
  Asset:              %(asset_tag)s
  FRU File ID:        %(fru_file_id)s
'''[1:-1] % area.__dict__

            except KeyError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding keys in product info area')

            try:
                if len(area.custom_mfg_info) != 0:
                    print '  Custom Board Info Records:'
                    for field in area.custom_mfg_info:
                        print '    %s' % field
            except AttributeError:
                syslog.syslog(syslog.LOG_WARNING,
                    'Error finding attributes in custom mfg info area (prod)')

        # TODO: When we have more time !
        # Multirecords
        # area = inv.multirecord_area
        # if area:
        #     print 'Multirecord Area:'
        #     for record in area.records:
        #         print '  %s' % record
