/* Copyright 2020 Cumulus Networks, Inc.  All rights reserved. */
#include <sys/file.h>
#include <dirent.h>
#include <sys/syscall.h>
#include <utime.h>
#include <string.h>
#include <libgen.h>
#include <execinfo.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include "log.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <errno.h>

#include "cs_mgr.h"
#include "cs_mgr_intf.h"

const char *CSMGR_STATE_FILE = CSMGR_STATE_FP;
const char *MLX_BOOT_MODE_FILE = MLX_SDK_BOOT_MODE_FILE;

/*
 * csmgr_next_boot_mode
 *
 * If this boot involved upgrade, subsequent restart will not be "upgrade". A controlled restart 
 * will update the file appropriately - this is a safeguard against uncontrolled restart.
 */

Mode
csmgr_next_boot_mode (Mode curr_mode) {
    switch(curr_mode) {
    case SYS_UPGRADE_REBOOT_COLD:
        return(REBOOT_COLD);
    case SYS_UPGRADE_REBOOT_FAST:
        return(REBOOT_FAST);
    case SYS_UPGRADE_REBOOT_WARM:
        return(REBOOT_WARM);
    default:
      return(curr_mode);
    }
}

bool
record_csmgr_state(Module module, Mode mode) {

	FILE *fp;
	char message[MAX_MODE_STR_LEN];

	sprintf(message, "%s" , mode_to_str(mode));

	fp = fopen(CSMGR_STATE_FILE , "w");
	if (!fp) {
		ERRLOG("Failed to open the csmgr state file to write the state");
		return false;
	}
	fprintf(fp, "%s", message);
	fclose(fp);
	return true;
}

/*
 * read_csmgr_state
 *
 * Reads the starting state for CSMgr. We have to account for CSMgrd restarts, someone rebooting
 * (or kernel crash) outside csmgrd infrastructure ("reboot").
 * The warm / fast boot are supported only on MLX platforms and mode is defined in MLX_BOOT_MODE_FILE
 */

Mode
read_csmgr_state(Module module) {

    FILE *fp;
    char state[MAX_MODE_STR_LEN];
    Mode mode, hw_mode, next_boot_mode;
    int v;

    fp = fopen(CSMGR_STATE_FILE , "r");
    
    if ((!fp) || (fgets(state, MAX_MODE_STR_LEN, fp) == NULL)) {
        mode = CSMGR_DEFAULT_BOOT_MODE;
	INFOLOG("No csmgr start state (%s) set to %s\n", strerror(errno), mode_to_str(mode));
    } else {
        mode = str_to_mode(state);
	if (mode == -1) {
	    mode = CSMGR_DEFAULT_BOOT_MODE;
	    ERRLOG("Invalid mode (%s) starting in %s\n", state,  mode_to_str(mode));
	}
    }
    
    if (fp) fclose(fp);

    fp = fopen(MLX_BOOT_MODE_FILE, "r");

    if ((!fp) || (fgets(state, MAX_MODE_STR_LEN, fp) == NULL)) {
	hw_mode = REBOOT_COLD;
    } else {
        v = atoi(state);
	hw_mode = (v == MLX_FAST_BOOT) ? REBOOT_FAST : REBOOT_COLD;
    }

    if (fp) fclose(fp);

    /* upgrade is true only for the immediate boot - clear for next round. */

    next_boot_mode = csmgr_next_boot_mode(mode);

    if (next_boot_mode != mode) {
        record_csmgr_state(module, next_boot_mode);
    }

    INFOLOG("Platform boot mode %s, CSMgr Requested Mode %s\n", mode_to_str(hw_mode), mode_to_str(mode));

    /* if hw_mode is FAST => we have gone through a CSMgr managed reboot => csmgr state file is good. */

    if (hw_mode == REBOOT_FAST) {
        return(mode);
    }

    /* MLX mode says cold boot => it WAS a cold boot. Is it post upgrade? */

    return((next_boot_mode == mode) ? REBOOT_COLD : SYS_UPGRADE_REBOOT_COLD);
}
