#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdbool.h>
#include <string.h>
#include <errno.h>
#include "cs_mgr_cli.h"

bool call_csmgr = true;

bool csmgr_process_upgrade(bool upgrade);

/*
 * csmgr_process_set_port_state
 *
 * Put all front panel ports down if up is false, restore configured port config if up is true.
 *
 * Note: The state are non persistent across reboots). The action is stored in CSMGR_MAINTENACE_PORT_FILE
 * created in /tnp => automatically removed on reboots. CSMgr uses the file only for the purpose of "show"
 * commands. There is no exchange with CSMgr.
 */

bool
csmgr_set_ports_state (bool up) {
    char cmd[CSMGR_CLI_MAX_BUFFER];

    if (up) {
        system("ifreload -a");
	sprintf(cmd, "rm -f %s", CSMGR_PORTS_IN_MAINTENACE_PORT_MODE_FILE);
    } else {
        system("ifdown -a -X eth0");
	sprintf(cmd, "touch %s", CSMGR_PORTS_IN_MAINTENACE_PORT_MODE_FILE);
    }

    system(cmd);

    return(false);
}

void show_usage(char *exec_name)
{
    printf("%s {-f | -c | -m<0|1> | -p<0|1> | -u | -d | -s | -h}\n", exec_name);
    //printf("-w: Warmboot\n");
    printf("-f: Fastboot\n");
    printf("-c: Coldboot\n");
    printf("-m<0|1>: Maintenance mode disable or enable\n");
    printf("-p<0|1>: disable or enable all configured ports\n");
    printf("-u: upgrade\n");
    printf("-d: upgrade dry-run\n");
    printf("-s: Show status\n");
    printf("-h: Show usage\n");
}

cs_mgr_cli_cmd_t parse_cmdline(int argc, char *argv[])
{
    cs_mgr_cli_cmd_t cmd = INVALID;
    char*            maint_op = NULL;
    char*            port_op = NULL;
    int              cmdline;

    if (argc != 2) {
        goto show_usage_goto;
    }

    if ((cmdline = getopt(argc, argv, "fcm:p:udsh")) == -1) {
        goto show_usage_goto;
    }

    switch (cmdline) {
    case 'w':
        cmd = WARMBOOT;
        break;

    case 'f':
        cmd = FASTBOOT;
        break;

    case 'c':
        cmd = COLDBOOT;
        break;

    case 'm':
        maint_op = optarg;
        if (strlen(maint_op) != 1) {
            goto show_usage_goto;
        }

        if (maint_op[0] == '0') {
            cmd = MAINTENANCE_MODE_DISABLE;
        } else if (maint_op[0] == '1') {
            cmd = MAINTENANCE_MODE_ENABLE;
        } else {
            goto show_usage_goto;
        }
        break;

    case 'p':
        port_op = optarg;
        if (strlen(port_op) != 1) {
            goto show_usage_goto;
        }

        if (port_op[0] == '0') {
	    cmd = MAINTENANCE_PORT_DOWN;
        } else if (port_op[0] == '1') {
	    cmd = MAINTENANCE_PORT_RESTORE;
        } else {
            goto show_usage_goto;
        }
	call_csmgr = csmgr_set_ports_state((cmd == MAINTENANCE_PORT_RESTORE) ? true : false);
        break;

    case 'u':
        cmd = UPGRADE_MODE;
        call_csmgr = csmgr_process_upgrade(true);
        break;

    case 'd':
        cmd = UPGRADE_DRY_RUN_MODE;
        call_csmgr = csmgr_process_upgrade(false);
        break;

    case 's':
        cmd = SHOW;
        break;

    default:
        goto show_usage_goto;
        break;
    }

    return cmd;

show_usage_goto:
    show_usage(argv[0]);
    return cmd;
}

bool
csmgr_handle_sock_ret (int fd, int err) {
    char *errmsg;
    int buf[10];  /* dummy buf. */
    
    if (err > 0) {
     return(true);
    }
    
    if (err < 0) {
      errmsg = "read error";
      goto  csmgr_handle_sock_ret_error;
    }

    /* err = 0 - exit if socket closed. */

    if (recv(fd, buf, sizeof(buf), MSG_PEEK | MSG_DONTWAIT) == 0) {
      errmsg = "server socket closed\n";
      goto csmgr_handle_sock_ret_error;
    }
    
    return(true);
    
 csmgr_handle_sock_ret_error:
    close(fd);
    perror(errmsg);
    exit(-1);    
}

bool
csmgr_process_upgrade(bool upgrade) {

    char *cmd = NULL;

    /* Dry run. */
    
    if (!upgrade) {
        cmd = "sudo apt-get update; sudo apt-get upgrade --dry-run";
        system(cmd);
        return(upgrade);
    }

    /* actual upgrade. */
    
    cmd = "sudo apt-get update";
    system(cmd);

    /* 
     * Store an md5 checksum of the apt history file. An apt upgrade if it actually upgrades components
     * will change the content. A different md5 will tell us some components were upgraded vs everthing
     * is already up to date. We just need to know if something got upgraded.
     */ 

    cmd = "md5sum /var/log/apt/history.log > /tmp/apt-hist.md5.before";
    system(cmd);

    /* do the upgrade - retain existing configuration files. */
    
    cmd = "sudo apt-get -y -o Dpkg::Options::=\"--force-confdef\" -o Dpkg::Options::=\"--force-confold\" upgrade";
    system(cmd);

    cmd = "md5sum /var/log/apt/history.log > /tmp/apt-hist.md5.after";
    system(cmd);

    cmd = "diff /tmp/apt-hist.md5.before /tmp/apt-hist.md5.after > /tmp/apt-hist.md5.diff";
    system(cmd);

    /* if the md5 is unchanged, the diff will be of zero size, */

    FILE *fp = fopen("/tmp/apt-hist.md5.diff", "r");

    if (!fp) {
      fprintf(stderr, "error in upgrade %s\n", strerror(errno));
      return(false);
    }

    fseek(fp, 0L, SEEK_END);
    long int res = ftell(fp);

    if (res == 0L) {
      printf(" No components were upgraded\n");
      fclose(fp);
      return(false);
    }
    
    fclose(fp);
    return(true);
}

int main(int argc, char* argv[])
{
    struct sockaddr_un addr;
    int                fd;
    int                ret;
    cs_mgr_cli_cmd_t   cmd;
    char resp_buf[CSMGR_CLI_MAX_BUFFER];
    cs_mgr_cli_msg_t   cli_msg;
    cs_mgr_cli_resp_t *cli_msg_resp = (cs_mgr_cli_resp_t *)resp_buf;
    char *platform = "platform-detect";
    char data[CSMGR_CLI_MAX_BUFFER];
    FILE *fp;

    /* Open the command for reading. */
    fp = popen(platform, "r");
    if (fp == NULL) {
        printf("Failed to detect the platform\n");
        return 1;
    }

    /* Read the output a line at a time */
    if (fgets(data, sizeof(data)-1, fp) != NULL) {
        if (!strstr(data, "mlnx")  && !strstr(data, "cumulus,vx")) {
           printf("Cumulus Smart Upgrade is not supported on %s\n", data);
	   pclose(fp);
           return 1;
        }
    }

    if (fp) pclose(fp);
    
    cmd = parse_cmdline(argc, argv);
    if (cmd == INVALID) {
        exit(-1);
    }

    if (!call_csmgr) {
        return 0;
    }

    if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
        perror("creating socket error");
        exit(-1);
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    strncpy(addr.sun_path, CSMGR_CLI_SOCK_PATH, sizeof(addr.sun_path) - 1);

    if (connect(fd, (struct sockaddr*) &addr, sizeof(addr)) == -1) {
        perror("error connecting to csmgrd");
        exit(-1);
    }

    if (cmd == FASTBOOT || cmd == WARMBOOT) {
        system("/usr/sbin/send_gratarp.sh");
    }

    /* 
     * Now send the cli command to csmgrd
     */
    memset(&cli_msg, 0, sizeof(cli_msg));
    cli_msg.cmd = cmd;
    ret = write(fd, &cli_msg, sizeof(cli_msg));
    if (ret < 0) {
        perror("write error");
        exit(-1);
    }

    /* block on a read from csmgrd (show may have continuos o/p) */

    while (1) {
      int read_so_far, total_len;
      ret = read(fd, cli_msg_resp, sizeof(*cli_msg_resp));
      csmgr_handle_sock_ret(fd, ret);   /* will exit on error. */

      if (cli_msg_resp->status.cmd == RESP_OVER) break;
      
      /* only other response is for SHOW (response is SHOW_IN_PROGRESS) */

      if ((cli_msg.cmd != SHOW) || (cli_msg_resp->status.cmd != SHOW_IN_PROGRESS)) {   
	sprintf(resp_buf, "unxpected response 0x%x to request 0x%x", cli_msg_resp->status.cmd, cli_msg.cmd);
	perror(resp_buf);
	break;
      }

      read_so_far = 0;
      total_len = cli_msg_resp->len - sizeof(*cli_msg_resp);  /* only the data length. */

      while ((total_len - read_so_far) > 0) {
	ret = read(fd, resp_buf + read_so_far, total_len - read_so_far); /* reuse resp_buf. */
	csmgr_handle_sock_ret(fd, ret);   /* will exit on error. */
	read_so_far += ret;
      }
      resp_buf[total_len] = '\0';
      printf("%s", resp_buf);
    }
    
    close(fd);

    return 0;
}
