#include <stdio.h>
#include <unistd.h>
#include <stdbool.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdlib.h>
#include <errno.h>
#include "log.h"
#include <pthread.h>

#include <cs_mgr.h>
#include <cs_mgr_intf.h>
#include <cs_mgr_cli.h>

char *csmgr_cli_socket_path = CSMGR_CLI_SOCK_PATH;
bool csmgr_cli_active = false;
pthread_t csmgr_cli_thread_id = 0;
int cli_fd = -1;   /* the initial socket fd fro CLI */
bool upgraded = false;    /* if we have upgraded the image in this cycle. */
time_t maintenance_start_time;

/*
 * csmgr_process_show
 *
 * Send streaming status to the ctl in displayable form. ctl will do the format conversion as needed.
 */

bool
csmgr_process_show (int conn_fd, cs_mgr_cli_msg_t *cli_msg, bool upgraded) {
    char resp_buf[CSMGR_CLI_MAX_BUFFER];
    cs_mgr_cli_resp_t *resp = (cs_mgr_cli_resp_t *)resp_buf;
    int mod;
    int active_mods = csmgr_get_num_active_modules();
    char *no_mod_response = " No Actively Registered Modules\n";
    int ret;

    resp->status.cmd = SHOW_IN_PROGRESS;

    if (!active_mods) {
        resp->len = sprintf(resp->data, "%s", no_mod_response) + sizeof(*resp);
	if ((ret = write(conn_fd, resp, resp->len)) != resp->len) {
	    ERRLOG("Error in null response to ctl %s\n", strerror(errno));
	}
	return(true);
    }

    resp->len = sprintf(resp->data, " Current System Mode: %s ", mode_to_str(operative_mode));

    if (operative_mode == MAINTENANCE) {
      if (maintenance_start_time) {
	  time_t time_now = time(NULL);
	  char *start_time = ctime(&maintenance_start_time);
	  start_time[strlen(start_time) - 1] = '\0';
	  time_t diff_time = difftime(time_now, maintenance_start_time);
	  long int sec = diff_time % 60, hr = diff_time/3600, mnt = (diff_time - hr * 3600)/60;

	  resp->len += sprintf(resp->data + resp->len, "since %s (Duration: %02ld:%02ld:%02ld)\n",
			       start_time, hr, mnt, sec);
      } else {
	  resp->len += sprintf(resp->data + resp->len, "%s\n", "(unknown duration)");
      }
    } else {
      resp->len += sprintf(resp->data + resp->len, "\n");
    }

    if (access(CSMGR_PORTS_IN_MAINTENACE_PORT_MODE_FILE, F_OK) == 0) {
        resp->len += sprintf(resp->data + resp->len, " Ports shutdown for Maintenance\n");
    }

    resp->len += sprintf(resp->data + resp->len, " Boot Mode: %s %s\n %d registered module%s\n",
			 mode_to_str(startup_operative_mode),
			 (upgraded) ? "(pending upgrade)" : " ", active_mods, (active_mods > 1) ? "s" : "");
    resp->len += sizeof(*resp);
    
    if ((ret = write(conn_fd, resp, resp->len)) != resp->len) {
        ERRLOG("Error in header write (len %d) to ctl %s\n", resp->len, strerror(errno));
	return(false);
    }
    
    for (mod = 2; mod < MAX_MODULES; mod++) {
        if (!csmgr_is_mod_registered(mod)) continue;
	resp->len = sprintf(resp->data, " %-8s: %s, %s\n", mod_id_to_str(mod),
			    mode_to_str(registered_mods[mod].mod_status.mode.mode),
			    mod_state_to_str(registered_mods[mod].mod_status.mode.state)) + sizeof(*resp);
	if ((ret = write(conn_fd, resp, resp->len)) != resp->len) {
	    ERRLOG("Error (%s) in write (len %d)  to ctl for mod %s\n", strerror(errno), resp->len,
		   mod_id_to_str(mod));
	    return(false);
	}
    }
    
    return(true);
}

/*
 * csmgr_process_upgrade
 *
 * Record the update state (the ctl will do the actual upgrade).
 * 
 */

bool
csmgr_process_upgrade (int conn_fd, cs_mgr_cli_msg_t *cli_msg, bool *upgraded) {

    *upgraded = record_csmgr_state(ALL_MODULES, UPGRADE);

    return(*upgraded);
}

/*
 * csmgr_process_cli
 *
 * Processes the command from the cli client and returns status to the caller. The upgrade (dry run and
 * install) will complete in the calling cli (csmgrctl).
 *
 * Note: The reboot option triggers a flush of pending i/o and temporarily brings down the mgmt port.
 * This safeguards the OS during reboot from stray packets from the mgmt interface.
 */

int 
csmgr_process_cli (int conn_fd, int len, void * cmd) {
    cs_mgr_cli_msg_t *cli_msg = (cs_mgr_cli_msg_t*) cmd;
    cs_mgr_cli_resp_t resp;
    int rc = 0;
    Module target_mod = ALL_MODULES;   /* for now. For component upgrade,  it will come as a part of cli. */
    bool up = false;    /* only maintenance disbale is true. */
    Mode mode = -1;
    char buf[64];
    bool reboot_needed = false;

    switch(cli_msg->cmd) {
        case WARMBOOT:
            mode = upgraded ? SYS_UPGRADE_REBOOT_WARM : REBOOT_WARM;
            sprintf(buf, "sync; ifdown mgmt; reboot --warm");
            reboot_needed = true;
            break;
        case FASTBOOT:
            mode = upgraded ? SYS_UPGRADE_REBOOT_FAST : REBOOT_FAST;
            sprintf(buf, "sync; ifdown mgmt; reboot --fast");
            reboot_needed = true;
            break;
        case COLDBOOT:
            mode = upgraded ? SYS_UPGRADE_REBOOT_COLD : REBOOT_COLD;
            sprintf(buf, "sync; reboot");
            reboot_needed = true;
            break;
        case MAINTENANCE_MODE_ENABLE:
	    if (operative_mode != MAINTENANCE) {
	      mode = MAINTENANCE;
	    }
            break;
        case MAINTENANCE_MODE_DISABLE:
	    if (operative_mode == MAINTENANCE) {
	      mode = MAINTENANCE;
	      up = true;
	    }
            break;
        case UPGRADE_MODE:
            (void)csmgr_process_upgrade(conn_fd, cli_msg, &upgraded);
            break;
        case SHOW:
            (void)csmgr_process_show(conn_fd, cli_msg, upgraded);
            break; 
        default:
            ERRLOG("Unknown command 0x%x\n", cli_msg->cmd);
    }

    /* 
     * Store time for entering Maintenance mode. In other cases, record for the next boot. Then let the
     * modules play it out. All the "reboot" options will result in a reboot (duh) - so we will not return.
     */

    if (mode != -1) {
        if(mode != MAINTENANCE) {
	    record_csmgr_state(ALL_MODULES, mode);
        } else {
	  if (!up) {
	    maintenance_start_time = time(NULL);
	  }
	}
	csmgr_admin_modules(mode, target_mod, up);
    }

    memset(&resp, 0, sizeof(resp));
    resp.status.cmd = RESP_OVER;
    resp.len = sizeof(resp);
    rc = write(conn_fd, &resp, sizeof(resp));
    if (rc != sizeof(resp)) {
        ERRLOG("error in send %s\n", strerror(errno));
    }

    if (reboot_needed) {
        INFOLOG("Rebooting with %s option\n", buf);
        system(buf);
    }

    return(true);
}

void
csmgr_cli_close_connect (int cli_sock) {
    csmgr_cli_active = false;
    (void)close(cli_sock); 
    (void)unlink(csmgr_cli_socket_path);
}

/*
 * csmgr_cli_open_connect
 *
 * Create the local socket for the cli client to connect.
 */

bool
csmgr_cli_open_connect (int *cli_sock) {
    int fd;
    struct sockaddr_un addr;

    *cli_sock = -1;

    if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
        ERRLOG("%s: error in creating cli socket %s\n", __FUNCTION__, strerror(errno));
        return(false);
    }

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    strncpy(addr.sun_path, csmgr_cli_socket_path, sizeof(addr.sun_path)-1);


    if (bind(fd, (struct sockaddr*)&addr, sizeof(addr)) == -1) {
        ERRLOG("%s: error in bind cli socket %s\n", __FUNCTION__, strerror(errno));
        close(fd);
        return(false);
    }

    if (listen(fd, 5) == -1) {
        ERRLOG("%s: error in listen cli socket %s\n", __FUNCTION__, strerror(errno));
        return(false);
    }

    *cli_sock = fd;

    return(true);
}

/*
 * csmgr_cli_thread
 *
 * The cli processing thread that waits for incoming requests from the cli client. Presently, the
 * commands except upgrade will run to completion before returning to the caller => only one 
 * command will be active at any time. It simplifies actions without much drawback since these 
 * are all privileged operator operations and we really do not expect multiple clis instances
 * running to adminster these critical functions.
 * 
 */

void *
csmgr_cli_thread (void *nothing) {
    int conn_fd = -1;
    int rc;
    char buf[CSMGR_CLI_MAX_BUFFER];

    /* clean up vestige of last run (if any). */

    (void)unlink(csmgr_cli_socket_path);

    if (csmgr_cli_open_connect(&cli_fd) != true) {
        return(NULL);
    }

    csmgr_cli_active = true;

    /* wait for connection, receive and process the command and wait again. */
    
    while (csmgr_cli_active) {
        if ((conn_fd = accept(cli_fd, NULL, NULL)) == -1) {
	    ERRLOG("%s: error in accept cli socket %s\n", __FUNCTION__, strerror(errno));
            sleep(1);
            continue;
        }

        rc = read(conn_fd, buf, sizeof(buf));

        if (rc == -1) {
	    ERRLOG("%s: error in read cli socket %s\n", __FUNCTION__, strerror(errno));
        } else {
	    csmgr_process_cli(conn_fd, rc, buf);
	}
	
        close(conn_fd);
    }

    (void)close(conn_fd);
    (void)close(cli_fd);
    (void)unlink(csmgr_cli_socket_path);

    return((void *)1);
}

int
csmgr_start_cli (void) {

    /* start the cli listener thread. */

    if (pthread_create(&csmgr_cli_thread_id, NULL, csmgr_cli_thread,  NULL)) {
        fprintf(stderr, "error in cli pthread creation %s\n", strerror(errno));
        return(false);
    }

    return(true);
}

int
csmgr_stop_cli (void) {
    csmgr_cli_active = false;
    (void)pthread_cancel(csmgr_cli_thread_id);
    pthread_join(csmgr_cli_thread_id, (void **)NULL);
    csmgr_cli_close_connect(cli_fd);
    return(true);
}
