#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <pthread.h>
#include <unistd.h>
#include "zmq.h"
#include "cs_mgr_intf.h"

bool sub_thread_active  = false;
pthread_t sub_thread = 0;
int num_mods_of_interest = 0;
Module mods_of_interest[MAX_MODULES];
INFR_CB module_cb;
pthread_mutex_t req_resp_lock; 

void *sub_context = NULL;
void *sub_socket = NULL;

void csmgr_destroy_sub_channel (void) {

    if (sub_socket) {
        zmq_close(sub_socket);
	sub_socket = NULL;
    }

    if (sub_context) {
        zmq_ctx_destroy(sub_context);
	sub_context = NULL;
    }
}

int csmgr_create_sub_channel (void) {
    int  i;

    if (sub_context || sub_socket) {
        csmgr_destroy_sub_channel();
    }

    sub_context = zmq_ctx_new();

    if (sub_context == NULL) {
        printf("failed sub context %s\n", zmq_strerror(errno));
        return(false);
    }

    sub_socket = zmq_socket(sub_context, ZMQ_SUB);

    if (sub_socket == NULL) {
        printf("failed socket %s %s\n", __FUNCTION__, zmq_strerror(errno));
        goto csmgr_create_sub_channel_err;
    }

    i = zmq_connect(sub_socket, CSMGR_PUB_SUB_CHAN);

    if (i != 0) {
        printf("failed connect %s\n", zmq_strerror(errno));
        goto csmgr_create_sub_channel_err;
    }

    i = zmq_setsockopt(sub_socket, ZMQ_SUBSCRIBE, "", 0);

    if (i != 0) {
        printf("failed setsockopt %s\n", zmq_strerror(errno));
        goto csmgr_create_sub_channel_err;
    }

    return(true);

 csmgr_create_sub_channel_err:

    csmgr_destroy_sub_channel();

    return(false);
}

void * csmgr_sub_thread (void *nothing) {
    int i;
    void *status = NULL;
 
    sub_thread_active = true;

    char buf[MAX_MSG_LEN];

    while (sub_thread_active) {
        i = zmq_recv(sub_socket, buf, sizeof(buf), 0); //ZMQ_DONTWAIT);

        if (i == -1) {
            printf("sub channel receive error %s\n", zmq_strerror(errno));
            sleep(1);
            continue;
        }

        /* filter and do callback. */

        if (module_cb) {
            i = module_cb(i, buf);
        }
    }

    status = (void *)1;   /* just non null. */

csmgr_sub_thread_done:

    return(status);
}

void *comm_context = NULL;
void *comm_socket = NULL;
int comm_socket_timeout = CSMGR_DEFAULT_COMM_SOCKET_TIMEOUT;
int comm_socket_num_retry = CSMGR_DEFAULT_COMM_SOCKET_RETRY;
bool comm_socket_reset_on_error = CSMGR_DEFAULT_COMM_RESET_ON_ERROR;

/*
 * csmgr_set_socket_channel_parms
 *
 * Sets the comm socket connection mode - timeout (ZMQ_RCVTIMEO), ignore req-resp mismatches
 * (ZMQ_REQ_RELAXED), no wait on close (ZMQ_LINGER).
 */

bool csmgr_set_comm_socket_parms (void *comm_sock, int timeout) {
    int val = 1;

    if (!comm_sock) {
        return(false);
    }
      
    zmq_setsockopt(comm_sock, ZMQ_RCVTIMEO, &timeout, sizeof(timeout)); 
    zmq_setsockopt(comm_sock, ZMQ_REQ_CORRELATE, &val, sizeof(val));
    zmq_setsockopt(comm_sock, ZMQ_REQ_RELAXED, &val, sizeof(val));
    val = 0;
    zmq_setsockopt(comm_sock, ZMQ_LINGER, &val, sizeof(val));

    return(true);
}

void csmgr_destroy_comm_channel (void) {

    if (comm_socket) {
        zmq_close(comm_socket);
        comm_socket = NULL;
    }

    if (comm_context) {
        zmq_ctx_destroy(comm_context);
	comm_context = NULL;
    }
}

int csmgr_create_comm_channel (void) {

    comm_context = zmq_ctx_new();

    if (comm_context == NULL) {
        printf("failed zmq_ctx_new %s\n", zmq_strerror(errno));
        return(false);
    }

    comm_socket = zmq_socket(comm_context, ZMQ_REQ);

    if (comm_socket == NULL) {
        printf("failed zmq_socket %s\n", zmq_strerror(errno));
        goto csmgr_create_comm_channel_err;
    }

    csmgr_set_comm_socket_parms(comm_socket, comm_socket_timeout);
    
    int rc = zmq_connect(comm_socket, CSMGR_COMM_CHAN);

    if (rc != 0) {
        printf("failed zmq_connect %s\n", zmq_strerror(errno));
        goto csmgr_create_comm_channel_err;
    }

    return(true);

csmgr_create_comm_channel_err:

    csmgr_destroy_comm_channel();

    return(false);
}

int csmgr_stop_channels (void) {

    sub_thread_active = false;

    if (sub_thread) {
        pthread_cancel(sub_thread);
	sub_thread_active = false;
	pthread_join(sub_thread, (void **)NULL);
        sub_thread = 0;
    }

    csmgr_destroy_sub_channel();
    csmgr_destroy_comm_channel();

    pthread_mutex_destroy(&req_resp_lock);

    return(true);
}

/*
 * csmgr_start_channels
 *
 * Starts the sub end of the pub-sub channel and the REQ end of REQ-REP comm channel. The creation needs to
 * be atomic - the sub receiver can try to use the REQ-REP channels before it is created (send blocks on the 
 * same mutex). 
 */

int csmgr_start_channels (void) {
    bool status = true;
  
    if (pthread_mutex_init(&req_resp_lock, NULL) != 0) { 
        printf("failed mutex init %s\n", strerror(errno));
	return(false);
    }

    pthread_mutex_lock(&req_resp_lock);

    /* open the REQ side of the REQ/REP pair. */

    if (!csmgr_create_comm_channel()) {
        status = false;
	goto csmgr_start_channels_over;
    }

    /* create sub channel, background thread for listening to csmgr publish events. */

    if (!csmgr_create_sub_channel()) {
        status = false;
	goto csmgr_start_channels_over;
    }

    if (pthread_create(&sub_thread, NULL, csmgr_sub_thread,  NULL)) {
        fprintf(stderr, "error in sub pthread creation %s\n", strerror(errno));
	status = false;
    }

 csmgr_start_channels_over:
    pthread_mutex_unlock(&req_resp_lock);

    return(status);
}

/*
 * csmgr_module_connect
 *
 * Internal calls to register / unregister the module.
 */

bool csmgr_module_connect (Module self, bool connect) {
    char buf[MAX_MSG_LEN];
    char resp[MAX_MSG_LEN];
    msg_pkg *m = (msg_pkg *)buf;
    msg *entry = (msg *)m->entry;
    module_connect *mod_reg = (module_connect *)entry->data;
    int send;
	
    entry->type = (connect) ? MODULE_REGISTER : MODULE_UNREGISTER;
    entry->len = sizeof(*entry) + sizeof(module_connect);
    
    m->total_len = sizeof(*m) + entry->len;

    mod_reg->mod = self;

    /* register / unregister (response is just an ack. */
    
    send = csmgr_send(self, m->total_len, m, sizeof(resp), resp);

    if (send == -1) {
        printf("failed to send connect %d : %s\n", connect, zmq_strerror(errno));
        return(false);
    }

    /* response is just an ack. */

    m = (msg_pkg *)resp;
    entry = (msg *)m->entry;

    if (entry->type != ACK) {
        printf("%s: expected ack, received 0x%x\n", __FUNCTION__, entry->type);
	return(false);
    }

    return(true);
}

/*
 * csmgr_set_comm_channel_parms
 *
 * Sets the timeout interval for read (in ms) for the REQ-REP (comm) channel, the number
 * of retries on error. If reset_on_error is set (true), the underlying channel is
 * reset aka reopened.
 *
 * Note: This MUST be called before csmgr_register_cb(). Otherwise, the paramters will take
 * effect only after the first reset. The timeout value obeys the semantics of zmq_setsockopt().
 */

void csmgr_set_comm_channel_parms (int timeout, int num_retry, bool reset_on_error) {
    comm_socket_timeout = timeout;
    comm_socket_num_retry = num_retry;
    comm_socket_reset_on_error = reset_on_error;
}

/*
 * csmgr_register_cb
 *
 * Register callback, create channels for communication (for send/recv messages). The 
 * channel creation is atomic - the sub channel should not use the REQ-REP comm channel before 
 * it is ready. 
 */

int csmgr_register_cb (Module self, int num_mods, Module mod_list[], INFR_CB mod_cb) {
    int status;

    if (num_mods > MAX_MODULES) {
        return(false);
    }

    if (!csmgr_start_channels()) {
        return(false);
    }

    num_mods_of_interest = num_mods;
    memcpy(mods_of_interest, mod_list, num_mods*sizeof(Module));

    module_cb = mod_cb;

    return(csmgr_module_connect(self, true));
}

/*
 * csmgr_unregister
 *
 * Unregister for the callbacks. 
 */

int csmgr_unregister (Module self) {

    csmgr_module_connect(self, false);
    (void) csmgr_stop_channels();
    module_cb = NULL;

    return(true);
}

/*
 * csmgr_send
 *
 * Sends the "req" to csmgr and receives the response in resp. It is the callers responsibility
 * to provide adequate buffer (upto MAX_MSG_LEN).
 *
 * Returns : 0 on success.
 */

int csmgr_send (Module self, int req_len, void *req, int resp_len, void *resp) {
    int i;
    int num_try = comm_socket_num_retry;
    int buf_len = resp_len;

    /* REQ - RESP socket is transaction oriented (cannot interleave transactions). */
    
    pthread_mutex_lock(&req_resp_lock);

    if ((i = zmq_send(comm_socket, req, req_len, ZMQ_DONTWAIT)) != req_len) {
        printf("Error: sending %d bytes, send %d (%s)\n", req_len, i, zmq_strerror(errno));
        i = -1;
	goto csmgr_send_over;
    }

    /* wait for the response with timeout. Retry if the response is not available. */

    while (num_try--) {
        i = zmq_recv(comm_socket, resp, resp_len, 0);

	if (i == 0) goto csmgr_send_over;

	if ((i == -1) && (errno == EAGAIN)) continue;
	
	break;
    }

    /* This is error. */
    
    printf("Error Recv %d (%s)", i, zmq_strerror(errno));
    
    if (comm_socket_reset_on_error) {
        csmgr_destroy_comm_channel();
	if (!csmgr_create_comm_channel()) {
	    printf("failed restart comm socket\n");
	}
    }
    
 csmgr_send_over:
    pthread_mutex_unlock(&req_resp_lock);
    
    return(i);
}
