/*
 * Copyright 2014,2015,2016,2017,2018 Cumulus Networks, Inc.
 * All rights reserved.
 *   Based on audisp-example.c by Steve Grubb <sgrubb@redhat.com>
 *     Copyright 2009 Red Hat Inc., Durham, North Carolina.
 *     All Rights Reserved.
 *
 *   TACACS+ work based on pam_tacplus.c
 *     Copyright (C) 2010, Pawel Krawczyk <pawel.krawczyk@hush.com> and
 *     Jeroen Nijhof <jeroen@jeroennijhof.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: olson@cumulusnetworks.com>
 *
 * This audisp plugin is used for TACACS+ accounting of commands
 * being run by users known to the TACACS+ servers
 * It uses libsimple_tacacct to communicate with the TACACS+ servers
 * It uses the same configuration file format as the libnss_tacplus
 * plugin (but uses the file /etc/audisp/audisp-tacplus.conf to
 * follow the audisp conventions).
 *
 * You can test it by running commands similar to:
 *   ausearch --start today --raw > test.log
 *   ./audisp-tacplus < test.log
 *
 * Excluding some init/destroy items you might need to add to main, the
 * event_handler function is the main place that you would modify to do
 * things specific to your plugin.
 *
 */

#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <libaudit.h>
#include <auparse.h>
#include <unistd.h>
#include <sys/stat.h>

#include <tacplus/libtac.h>
#include <tacplus/map_tacplus_user.h>

#define _VMAJ 1
#define _VMIN 0
#define _VPATCH 0

#define MAX_INC 8 /* max nesting depth for include directive */

/* Global Data */
static volatile int stop = 0;
static volatile int reparse_cfg = 0;
static auparse_state_t *au = NULL;
static unsigned connected_ok;
static char *cfile_name[MAX_INC];
static FILE *cfile_stdio[MAX_INC];
time_t cfile_mtime[MAX_INC];

char *configfile = "/etc/audisp/audisp-tac_plus.conf";

/* Local declarations */
static void handle_event(auparse_state_t *au,
		auparse_cb_event_t cb_event_type, void *user_data);

typedef struct {
    struct addrinfo *addr;
    char *key;
    unsigned not_resp;
} tacplus_server_t;

/* set from configuration file parsing */
static tacplus_server_t tac_srv[TAC_PLUS_MAXSERVERS];
static int tac_srv_no, tac_key_no;
static char tac_service[64];
static char tac_protocol[64];
static char vrfname[64];
static int debug = 0;
static int acct_all; /* send accounting to all servers, not just 1st */
static struct sockaddr src_sockaddr;
static struct addrinfo src_addr_info;
static struct addrinfo *src_addr;

static const char *progname = "audisp-tacplus"; /* for syslogs and errors */


/*
 * SIGTERM handler
 */
static void
term_handler(int sig __attribute__ ((unused)))
{
        stop = 1;
}

/*
 * SIGHUP handler: re-read config
 */
static void
reparse_cfg_handler(int sig __attribute__ ((unused)))
{
        reparse_cfg = 1;
}

static void
chk_cfiles(void)
{
    int i;
    struct stat st;

    for (i=0; i<MAX_INC; i++) {
        if (cfile_name[i] && (stat(cfile_name[i], &st) == -1 ||
            st.st_mtime > cfile_mtime[i])) {
            reparse_cfg = 1;
            if (debug)
                syslog(LOG_DEBUG, "%s: config file %s was modified",
                       progname, cfile_name[i]);
            break;
        }
    }
}


/* Convert ip address string to address info.
 * It returns 0 on success, or -1 otherwise
 * It supports ipv4 only.
 */
int str_to_ipv4 (const char *srcaddr, struct addrinfo *p_addr_info)
{
    struct sockaddr_in *s_in;

    s_in = (struct sockaddr_in *)p_addr_info->ai_addr;
    s_in->sin_family = AF_INET;
    s_in->sin_addr.s_addr = INADDR_ANY;

    if (inet_pton(AF_INET, srcaddr, &(s_in->sin_addr)) == 1) {
        p_addr_info->ai_family = AF_INET;
        p_addr_info->ai_addrlen = sizeof (struct sockaddr_in);
        return 0;
    }
    return -1;
}

static void
audisp_tacplus_config(char *cfile, int level, int reinit)
{
    FILE *conf;
    char lbuf[256];
    static int complained;

    conf = fopen(cfile, "r");
    if(conf == NULL && !complained)
        syslog(LOG_WARNING, "%s: can't open config file %s: %m", progname,
               cfile);

    if (level >= MAX_INC)
        syslog(LOG_WARNING, "%s: config include nesting depth %d, not tracking"
               " %s", progname, level, cfile);
    else {
        if(cfile_name[level]) {
            if (cfile_stdio[level])
                fclose(cfile_stdio[level]);
             free(cfile_name[level]);
        }
        /*  save it whether it can be opened or not, so we can stat later */
        cfile_name[level] = strdup(cfile);
        if (!cfile_name[level])
                syslog(LOG_WARNING, "%s: Memory alloc failure for %s file, not"
                       " tracking: %m", progname, cfile);

        cfile_stdio[level] = conf;
        if (conf) {
            struct stat st;
            if (fstat(fileno(conf), &st) == -1) {
                syslog(LOG_WARNING, "%s: Failed %s file info, not tracking",
                       progname, cfile);
            }
            else
                cfile_mtime[level] = st.st_mtime;
        }
        else {
            /*  init for error cases */
            cfile_mtime[level] = 0;
        }
    }

    if(conf == NULL)
        return;

    while(fgets(lbuf, sizeof lbuf, conf)) {
        if(*lbuf == '#' || isspace(*lbuf))
            continue; /* skip comments, white space lines, etc. */
        strtok(lbuf, " \t\n\r\f"); /* terminate buffer at first whitespace */
        if(!strncmp(lbuf, "include=", 8)) {
            /*
             * allow include files, useful for centralizing tacacs
             * server IP address and secret.
             */
            if(lbuf[8]) /* else treat as empty config */
                audisp_tacplus_config(&lbuf[8], level+1, 0);
        }
        else if(!strncmp(lbuf, "debug=", 6))
            debug = strtoul(lbuf+6, NULL, 0);
        else if(!strncmp(lbuf, "acct_all=", 9))
            acct_all = strtoul(lbuf+9, NULL, 0);
        else if(!strncmp(lbuf, "vrf=", 4))
            tac_xstrcpy(vrfname, lbuf + 4, sizeof(vrfname));
        else if(!strncmp(lbuf, "service=", 8))
            tac_xstrcpy(tac_service, lbuf + 8, sizeof(tac_service));
        else if(!strncmp(lbuf, "protocol=", 9))
            tac_xstrcpy(tac_protocol, lbuf + 9, sizeof(tac_protocol));
        else if(!strncmp(lbuf, "login=", 6))
            tac_xstrcpy(tac_login, lbuf + 6, sizeof(tac_login));
        else if (!strncmp (lbuf, "timeout=", 8)) {
            tac_timeout = (int)strtoul(lbuf+8, NULL, 0);
            if (tac_timeout < 0) /* explict neg values disable poll() use */
                tac_timeout = 0;
            else /* poll() only used if timeout is explictly set */
                tac_readtimeout_enable = 1;
        }
        else if(!strncmp(lbuf, "secret=", 7)) {
            int i;
            /* no need to complain if too many on this one */
            if(tac_key_no < TAC_PLUS_MAXSERVERS) {
                if((tac_srv[tac_key_no].key = strdup(lbuf+7)))
                    tac_key_no++;
                else
                    syslog(LOG_ERR, "%s: unable to copy server secret %s",
                        progname, lbuf+7);
            }
            /* handle case where 'secret=' was given after a 'server='
             * parameter, fill in the current secret */
            for(i = tac_srv_no-1; i >= 0; i--) {
                if (tac_srv[i].key)
                    continue;
                tac_srv[i].key = strdup(lbuf+7);
            }
        }
        else if(!strncmp(lbuf, "server=", 7)) {
            if(tac_srv_no < TAC_PLUS_MAXSERVERS) {
                struct addrinfo hints, *servers, *server;
                int rv;
                char *port, server_buf[sizeof lbuf];

                memset(&hints, 0, sizeof hints);
                hints.ai_family = AF_UNSPEC;  /* use IPv4 or IPv6, whichever */
                hints.ai_socktype = SOCK_STREAM;

                strcpy(server_buf, lbuf + 7);

                port = strchr(server_buf, ':');
                if(port != NULL) {
                    *port = '\0';
					port++;
                }
                if((rv = getaddrinfo(server_buf, (port == NULL) ?
                            "49" : port, &hints, &servers)) == 0) {
                    for(server = servers; server != NULL &&
                        tac_srv_no < TAC_PLUS_MAXSERVERS;
                        server = server->ai_next) {
                        tac_srv[tac_srv_no].addr = server;
                        /* use current key, if our index not yet set */
                        if(tac_key_no && !tac_srv[tac_srv_no].key)
                            tac_srv[tac_srv_no].key =
                                strdup(tac_srv[tac_key_no-1].key);
                        tac_srv_no++;
                    }
                }
                else {
                    syslog(LOG_ERR,
                        "%s: skip invalid server: %s (getaddrinfo: %s)",
                        progname, server_buf, gai_strerror(rv));
                }
            }
            else {
                syslog(LOG_ERR, "%s: maximum number of servers (%d) exceeded, "
                    "skipping", progname, TAC_PLUS_MAXSERVERS);
            }
        }
        else if (!strncmp (lbuf, "source_ip=", 10)) {
            const char *srcip = lbuf + 10;
            /* if source ip address, convert it to addr info  */
            memset (&src_addr_info, 0, sizeof (struct addrinfo));
            memset (&src_sockaddr, 0, sizeof (struct sockaddr));
            src_addr_info.ai_addr = &src_sockaddr;
            if (str_to_ipv4 (srcip, &src_addr_info) == 0)
                src_addr = &src_addr_info;
            else {
                src_addr = NULL; /* for re-parsing or errors */
                syslog(LOG_WARNING,
                       "%s: unable to convert %s to an IPv4 address", progname,
                       lbuf);
            }
        }
        else if(debug) { /* ignore unrecognized lines, unless debug on */
            /*  Don't complain about user_homedir, we don't use it,
             *  but commonly set.
             */
            if(strncmp(lbuf, "user_homedir=", 13))
                syslog(LOG_WARNING, "%s: unrecognized parameter: %s",
                    progname, lbuf);
        }
    }

    if(level == 0) {
        if (!tac_service[0])
            strcpy(tac_service, "shell");

        if (tac_srv_no == 0) {
            if (!complained) {
                syslog(LOG_ERR, "%s version %d.%d.%d: No servers defined in"
                       " configuration files.", progname, _VMAJ, _VMIN,
		       _VPATCH);
                complained = 1;
            }
        }
        else {
            if(reinit)
                syslog(LOG_NOTICE, "%s re-initialized configuration", progname);
            complained = 0;

            if(debug) {
                int n;
                syslog(LOG_NOTICE, "%s version %d.%d.%d tacacs service=%s",
                       progname, _VMAJ, _VMIN, _VPATCH, tac_service);

                for(n = 0; n < tac_srv_no; n++)
                    syslog(LOG_DEBUG, "%s: tacacs server[%d] { addr=%s,"
                           " key='%s' }", progname, n,
                           tac_ntop(tac_srv[n].addr->ai_addr), tac_srv[n].key);
            }
        }
    }
}


static void
reload_config(int reload)
{
    int i, nservers;

    reparse_cfg = 0;

    /*  reset the config variables that we use, freeing memory where needed */
    nservers = tac_srv_no;
    tac_srv_no = 0;
    tac_key_no = 0;
    vrfname[0] = '\0';
    tac_service[0] = '\0';
    tac_protocol[0] = '\0';
    tac_login[0] = '\0';
    debug = 0;
    acct_all = 0;
    tac_timeout = 0;
    src_addr = NULL;

    for(i = 0; i < nservers; i++) {
        if(tac_srv[i].key) {
            free(tac_srv[i].key);
            tac_srv[i].key = NULL;
        }
        tac_srv[i].addr = NULL;
        tac_srv[i].not_resp = 0;
    }

    connected_ok = 0; /*  reset connected state (for possible vrf) */

    audisp_tacplus_config(configfile, 0, reload);
}

int
main(int argc, char *argv[])
{
	char tmp[MAX_AUDIT_MESSAGE_LENGTH+1];
	struct sigaction sa;

	/* if there is an argument, it is an alternate configuration file */
	if(argc > 1)
		configfile = argv[1];

	reload_config(0);

	/* Register sighandlers */
	sa.sa_flags = 0;
	sigemptyset(&sa.sa_mask);
	/* Set handler for the ones we care about */
	sa.sa_handler = term_handler;
	sigaction(SIGTERM, &sa, NULL);

        sa.sa_flags = SA_RESTART;
	sa.sa_handler = reparse_cfg_handler;
	sigaction(SIGHUP, &sa, NULL);

	/* Initialize the auparse library */
	au = auparse_init(AUSOURCE_FEED, 0);
	if(au == NULL) {
		syslog(LOG_ERR, "%s: exitting due to auparse init errors",
                       progname);
		return -1;
	}
	auparse_add_callback(au, handle_event, NULL, NULL);
	do {
		/*
		 * Now the event loop.  For some reason, audisp doesn't send
		 * us the ANOM_ABEND until flushed by another event. and it
		 * therefore has the timestamp of the next event.  I can't find
		 * any parameters to affect that.
		 */
		while(fgets(tmp, MAX_AUDIT_MESSAGE_LENGTH, stdin) && stop==0) {
			/*
			 * Any changes via stat or HUP?
			 * If so, we need to reparse
			 */
			chk_cfiles();
			if(reparse_cfg)
				reload_config(1);

			auparse_feed(au, tmp, strnlen(tmp,
						MAX_AUDIT_MESSAGE_LENGTH));
		}
		if(feof(stdin))
			break;
	} while(stop == 0);

	/* Flush any accumulated events from queue */
	auparse_flush_feed(au);
	auparse_destroy(au);

	return 0;
}

int
send_acct_msg(int tac_fd, int type, char *user, char *tty, char *host,
    char *cmd, uint16_t taskid)
{
    char buf[128];
    struct tac_attrib *attr;
    int retval;
    struct areply re;

    attr=(struct tac_attrib *)tac_xcalloc(1, sizeof(struct tac_attrib));

    snprintf(buf, sizeof buf, "%lu", (unsigned long)time(NULL));
    tac_add_attrib(&attr, "start_time", buf);

    snprintf(buf, sizeof buf, "%hu", taskid);
    tac_add_attrib(&attr, "task_id", buf);

    tac_add_attrib(&attr, "service", tac_service);
    if(tac_protocol[0])
      tac_add_attrib(&attr, "protocol", tac_protocol);
    tac_add_attrib(&attr, "cmd", (char*)cmd);

    re.msg = NULL;
    retval = tac_acct_send(tac_fd, type, user, tty, host, attr);

    if(retval < 0)
        syslog(LOG_WARNING, "%s: send of accounting msg failed: %m", progname);
    else if(tac_acct_read(tac_fd, &re) != TAC_PLUS_ACCT_STATUS_SUCCESS ) {
        syslog(LOG_WARNING, "%s: accounting msg response failed: %m", progname);
        retval = -1;
    }

    tac_free_attrib(&attr);
    if(re.msg != NULL)
        free(re.msg);

    return retval >= 0 ? 0 : 1;
}

/*
 * If all servers have been unresponsive, clear that state, so we try
 * them all.  It might have been transient.
 * Also, after 10 minutes have passed, clear the non-responding state,
 * so that we start using the servers again for acct_all=1 case, if
 * they are down and then come back.
 */
static void tac_chk_anyresp(void)
{
    int i, anyok=0;
    static time_t next_force;
    time_t now;

    now = time(NULL);
    for(i = 0; i < tac_srv_no; i++) {
        if (!tac_srv[i].not_resp)
            anyok++;
    }
    if (!anyok || (anyok != tac_srv_no && now > next_force)) {
        next_force = time(NULL) + 10 * 60;
        for(i = 0; i < tac_srv_no; i++)
            tac_srv[i].not_resp = 0;
    }
}


/*
 * Send the accounting record to the TACACS+ server.
 *
 * We have to make a new connection each time, because libtac is single threaded
 * (doesn't support multiple connects at the same time due to use of globals)),
 * and doesn't have support for persistent connections.
 */
static void
send_tacacs_acct(char *user, char *tty, char *host, char *cmdmsg, int type,
    uint16_t task_id)
{
    int retval, srv_i, srv_fd;

    tac_chk_anyresp();
    for(srv_i = 0; srv_i < tac_srv_no; srv_i++) {
        if (tac_srv[srv_i].not_resp)
            continue; /*  don't retry if previously not responding */
        srv_fd = tac_connect_single(tac_srv[srv_i].addr, tac_srv[srv_i].key,
            src_addr, vrfname[0]?vrfname:NULL);
        if(srv_fd < 0) {
            tac_srv[srv_i].not_resp = 1;
            syslog(LOG_WARNING, "%s: connection to %s failed (%d) to send"
                " accounting record: %m", progname,
                tac_ntop(tac_srv[srv_i].addr->ai_addr), srv_fd);
            continue;
        }
        retval = send_acct_msg(srv_fd, type, user, tty, host, cmdmsg, task_id);
        if(retval)
            syslog(LOG_WARNING, "%s: error sending accounting record to %s: %m",
                progname, tac_ntop(tac_srv[srv_i].addr->ai_addr));
        close(srv_fd);
        if(!retval) {
            connected_ok = 1;
            if(!acct_all)
                break; /* only send to first responding server */
        }
    }
}

/*
 * encapsulate the field lookup, and rewind if needed,
 * rather than repeating at each call.
 */
static const char *get_field(auparse_state_t *au, const char *field)
{
    const char *str;
    if(!(str=auparse_find_field(au, field))) {
        auparse_first_field(au);
        if(!(str=auparse_find_field(au, field))) {
            /* sometimes auparse_first_field() isn't enough, depending
             * on earlier lookup order. */
            auparse_first_record(au);
            if(!(str=auparse_find_field(au, field)))
                return NULL;
        }
    }
    return str;
}

/* find an audit field, and return the value for numeric fields.
 * return 1 if OK (field found and is numeric), otherwise 0.
 * It is somewhat smart, in that it will try from current position in case code
 * is written "knowing" field order; if not found will rewind and try again.
 */
static unsigned long
get_auval(auparse_state_t *au, const char *field, int *val)
{
    int rv;

    if(!get_field(au, field))
        return 0;
    rv = auparse_get_field_int(au);
    if(rv == -1 && errno)
        return 0;
    *val = rv;
    return 1;
}


/*
 * Get the audit record for exec system calls, and send it off to the
 * tacacs+ server.   Lookup the original tacacs username first.
 * This just gets us the starts of commands, not the stop, which would
 * require matching up the exit system call.   For now, don't bother.
 * Maybe add some caching of usernames at some point.
 * Both auid and sessionid have to be valid for us to do accounting.
 * We don't bother with really long cmd names or really long arg lists,
 * we stop at 240 characters, because the longest field tacacs+ can handle
 * is 255 characters, and some of the accounting doesn't seem to work
 * if right at full length.
 */
static void get_acct_record(auparse_state_t *au, int type)
{
    int val, i, llen, tlen, freeloguser=0;
    int acct_type;
    pid_t pid;
    uint16_t taskno;
    unsigned argc=0, session=0, auid;
    char *auser = NULL, *loguser, *tty = NULL, *host = NULL;
    char *cmd = NULL, *ausyscall = NULL;
    char logbuf[240], *logptr, *logbase;

    if(get_field(au, "syscall"))
        ausyscall = (char *)auparse_interpret_field(au);

    /* exec calls are START of commands, exit (including exit_group) are STOP */
    if(ausyscall && !strncmp(ausyscall, "exec", 4)) {
        acct_type = TAC_PLUS_ACCT_FLAG_START;
    }
    else if(ausyscall && !strncmp(ausyscall, "exit", 4)) {
        acct_type = TAC_PLUS_ACCT_FLAG_STOP;
    }
    else if(type == AUDIT_ANOM_ABEND) {
        acct_type = TAC_PLUS_ACCT_FLAG_STOP;
    }
    else /* not a system call we care about */
        return;

    auid = session = val = 0;
    if(get_auval(au, "auid", &val))
        auid = (unsigned)val;
    if(auid == 0 || auid == (unsigned)-1) {
        /* we have to have auid for tacplus mapping */
        return;
    }
    if(get_auval(au, "ses", &val))
        session = (unsigned)val;
    if(session == 0 || session == (unsigned)-1) {
        /* we have to have session for tacplus mapping */
        return;
    }
    if(get_auval(au, "pid", &val)) {
        /*
         * Use pid so start and stop have matching taskno.  If pids wrap
         * in 16 bit space, we might have a collsion, but that's unlikely,
         * and with 16 bits, it could happen no matter what we do.
         */
        pid = (pid_t)val;
        taskno = (uint16_t) pid;
    }
    else /* should never happen, if it does, records won't match */
        taskno = tac_magic();

    if(get_field(au, "auid")) {
        auser = (char *)auparse_interpret_field(au);
    }
    if(!auser) {
        auser="unknown";
    }
    if(get_field(au, "tty"))
        tty = (char *)auparse_interpret_field(au);

    auparse_first_field(au);

    /*
     * pass NULL as the name lookup because we must have an auid and session
     * match in order to qualify as a tacacs session accounting record.  With
     * the NSS library, the username in auser will likely already be the login
     * name.
     */
    loguser = lookup_logname(NULL, auid, session, &host, NULL);
    if(!loguser) {
        char *user = NULL;

        if(auser) {
            user = auser;
        }
        else {
            auparse_first_field(au);
            if(auparse_find_field(au, "uid")) {
                user = (char *)auparse_interpret_field(au);
            }
        }
        if(!user)
            return; /* must be an invalid record */
        loguser = user;
    }
    else {
        freeloguser = 1;
    }

    if(get_field(au, "exe"))
        cmd = (char *)auparse_interpret_field(au);
    if(get_auval(au, "argc", &val))
        argc = (int)val;

    /*
     * could also grab "exe", since it can in theory
     * be different, and besides gives full path, so not ambiguous,
     * but not for now.
     */
    logbase = logptr = logbuf;
    tlen =  0;

    if(cmd) {
        i = 1; /* don't need argv[0], show full executable */
        llen = snprintf(logptr, sizeof logbuf - tlen, "%s", cmd);
        if(llen >= sizeof logbuf) {
            llen = sizeof logbuf - 1;
        }
        logptr += llen;
        tlen = llen;
    }
    else
        i = 0; /* show argv[0] */
    for(; i<argc && tlen < sizeof logbuf; i++) {
        char anum[13];
        snprintf(anum, sizeof anum, "a%u", i);
        if(get_field(au, anum)) { /* should always be true */
            llen = snprintf(logptr, sizeof logbuf - tlen,
                "%s%s", i?" ":"", auparse_interpret_field(au));
            if(llen >= (sizeof logbuf - tlen))
                break;
            logptr += llen;
            tlen += llen;
        }
    }

    /*
     * Put exit status after command name, the argument to exit is in a0
     * for exit syscall; duplicates part of arg loop below
     * This won't ever happen for processes that terminate on signals,
     * including SIGSEGV, unfortunately.  ANOM_ABEND would be perfect,
     * but it doesn't always happen, at least in jessie.
     */
    if(acct_type == TAC_PLUS_ACCT_FLAG_STOP && argc == 0) {
        if(get_field(au, "a0")) {
            (void)snprintf(logptr, sizeof logbuf - tlen,
                " exit=%d", auparse_get_field_int(au));
        }
        else if(get_auval(au, "sig", &val)) {
            (void)snprintf(logptr, sizeof logbuf - tlen,
                " exitsig=%d", (int)val);
        }
    }

    /*
     * loguser is always set, we bail if not.  For ANOM_ABEND, tty may be
     *  unknown, and in some cases, host may be not be set.
     */
    send_tacacs_acct(loguser, tty?tty:"UNK", host?host:"UNK", logbase, acct_type, taskno);

    if(host)
        free(host);

    if(freeloguser)
        free(loguser);
}

/*
 * This function receives a single complete event at a time from the auparse
 * library. This is where the main analysis code would be added.
 */
static void
handle_event(auparse_state_t *au, auparse_cb_event_t cb_event_type,
             void *user_data __attribute__ ((unused)))
{
    int type, num=0;

    if(cb_event_type != AUPARSE_CB_EVENT_READY) {
	    return;
    }

    /* Loop through the records in the event looking for one to process.
     * We use physical record number because we may search around and
     * move the cursor accidentally skipping a record.
     */
    while(auparse_goto_record_num(au, num) > 0) {
	type = auparse_get_type(au);
	/*
	 * we are only writing TACACS account records for syslog exec
	 * records.  login, etc. are handled through pam_tacplus
	 */
	switch(type) {
	    case AUDIT_SYSCALL:
	    case AUDIT_ANOM_ABEND:
		get_acct_record(au, type);
		break;
	    default:
		// for doublechecking dump_whole_record(au);
		break;
	}
	num++;
    }
}
